var _ = require('./utils');
var Promise = require('es6-promise').Promise;
var Remote = require('./remote');
var Prefetch = require('./prefetch');
var tokenizers = require('./tokenizers');
var oParser = require('./options_parser');
var SearchIndex = require('./search_index');
var Transport = require('./transport');

function Bloodhound(o) {
  o = oParser(o);

  this.sorter = o.sorter;
  this.identify = o.identify;
  this.sufficient = o.sufficient;

  this.local = o.local;
  this.remote = o.remote ? new Remote(o.remote) : null;
  this.prefetch = o.prefetch ? new Prefetch(o.prefetch) : null;

  // the backing data structure used for fast pattern matching
  this.index = new SearchIndex({
    identify: this.identify,
    datumTokenizer: o.datumTokenizer,
    queryTokenizer: o.queryTokenizer
  });

  // hold off on intialization if the intialize option was explicitly false
  o.initialize !== false && this.initialize();
}

Bloodhound.tokenizers = tokenizers;

_.mixin(Bloodhound.prototype, {

  // ### super secret stuff used for integration with jquery plugin

  __ttAdapter: function ttAdapter() {
    var that = this;

    return this.remote ? withAsync : withoutAsync;

    function withAsync(query, sync, async) {
      return that.search(query, sync, async);
    }

    function withoutAsync(query, sync) {
      return that.search(query, sync);
    }
  },

  _loadPrefetch: function loadPrefetch() {
    var that = this, promise, serialized;

    if (!this.prefetch) {
      return new Promise(function(resolve, reject) {
        resolve();
      });
    }

    else if (serialized = this.prefetch.fromCache()) {
      this.index.bootstrap(serialized);
      return new Promise(function(resolve, reject) {
        resolve();
      });
    }

    else {
      // this.prefetch.fromNetwork(done);
      return new Promise(function(resolve, reject) {

        // todo: promise
        that.prefetch.fromNetwork(function(err, data) {
          if (err) return reject(err);

          try {
            that.add(data);
            that.prefetch.store(that.index.serialize());
            resolve();
          } catch(e) {
            reject(e);
          }
        });
      });
    }
  },

  _initialize: function() {
    var that = this, deferred;

    // in case this is a reinitialization, clear previous data
    this.clear();

    (this.initPromise = this._loadPrefetch())
    .then(addLocalToIndex); // local must be added to index after prefetch

    return this.initPromise;

    function addLocalToIndex() { that.add(that.local); }
  },

  // ### public

  initialize: function(force) {
    return !this.initPromise || force ? this._initialize() : this.initPromise;
  },

  // TODO: before initialize what happens?
  add: function(data) {
    this.index.add(data);
    return this;
  },

  get: function(ids) {
    ids = _.isArray(ids) ? ids : [].slice.call(arguments);
    return this.index.get(ids);
  },

  search: function(query, sync, async) {
    var that = this, local;

    local = this.sorter(this.index.search(query));

    // return a copy to guarantee no changes within this scope
    // as this array will get used when processing the remote results
    sync(this.remote ? local.slice() : local);

    if (this.remote && local.length < this.sufficient) {
      this.remote.get(query, processRemote);
    }

    else if (this.remote) {
      // #149: prevents outdated rate-limited requests from being sent
      this.remote.cancelLastRequest();
    }

    return this;

    function processRemote(remote) {
      var nonDuplicates = [];

      // exclude duplicates
      _.each(remote, function(r) {
         !_.some(local, function(l) {
          return that.identify(r) === that.identify(l);
        }) && nonDuplicates.push(r);
      });

      async && async(nonDuplicates);
    }
  },

  all: function() {
    return this.index.all();
  },

  clear: function() {
    this.index.reset();
    return this;
  },

  clearPrefetchCache: function() {
    this.prefetch && this.prefetch.clear();
    return this;
  },

  clearRemoteCache: function() {
    Transport.resetCache();
    return this;
  },

  // DEPRECATED: will be removed in v1
  ttAdapter: function() {
    return this.__ttAdapter();
  }
});

module.exports = Bloodhound;
